<?php
/*********************************
 * export_ipath_services_csv.php
 * Streams CSV of ipath services using a JSON column map.
 *
 *
 * /* Examples of advanced formats you may enable later:
 * { "header": "From → To", "selector": "", "visible": false,
 * "fmt": { "type": "concat", "template": "{from_display_name} → {to_display_name}" } },
 *
 * { "header": "Plane Code", "selector": "routing_plane_name", "visible": false,
 * "fmt": { "type": "map", "map": { "Video": "VID", "Audio": "AUD", "Anc": "ANC" }, "fallback": "UNK" } }
 * /
 *
 *
 *
 *
 *
 *********************************/
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '/tmp/liveconn.log');
error_reporting(E_ALL);

header('Content-Type: text/csv; charset=utf-8');

$root = dirname(__DIR__);
$mapFile = $root . '/config/ipath_export_columns.json';

// Build URL to your existing page API and force limit=0
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
$selfBase = $scheme . '://' . $host;

// Pass-through filters (q, inst, rp, etc.)
$query = $_GET;
$query['limit'] = '0';
if (!isset($query['offset'])) $query['offset'] = '0';

$apiPath = '/hive/live-connections/api/get_ipath_services_page.php';
$apiUrl  = $selfBase . $apiPath . '?' . http_build_query($query);

// Load column map
$cols = json_decode(@file_get_contents($mapFile), true);
if (!is_array($cols)) {
    http_response_code(500);
    echo "Invalid column map JSON";
    exit;
}

// Fetch data
$resp = @file_get_contents($apiUrl);
if ($resp === false) {
    http_response_code(502);
    echo "Failed to fetch data";
    exit;
}
$json = json_decode($resp, true);
$rows = $json['rows'] ?? [];
if (!is_array($rows)) $rows = [];

// Prepare CSV output
$filename = 'ipath_services_' . date('Ymd_His') . '.csv';
header('Content-Disposition: attachment; filename="' . $filename . '"');

// Open output
$out = fopen('php://output', 'w');
if ($out === false) {
    http_response_code(500);
    echo "Cannot open output";
    exit;
}

// Helpers
function getByPath($arr, $path) {
    if ($path === '' || $path === null) return null;
    $parts = explode('.', $path);
    $v = $arr;
    foreach ($parts as $p) {
        if (is_array($v) && array_key_exists($p, $v)) {
            $v = $v[$p];
        } else {
            return null;
        }
    }
    return $v;
}

function formatValue($val, $row, $fmt) {
    if (!$fmt || !is_array($fmt) || empty($fmt['type'])) {
        return is_scalar($val) ? $val : (is_null($val) ? '' : json_encode($val, JSON_UNESCAPED_SLASHES));
    }
    switch ($fmt['type']) {
        case 'bool':
            $truthy = $fmt['truthy'] ?? 'Yes';
            $falsy  = $fmt['falsy']  ?? 'No';
            $isTrue = ($val === true) || $val === 1 || $val === '1' || $val === 'true' || $val === 't' || $val === 'Y' || $val === 'yes';
            return $isTrue ? $truthy : $falsy;

        case 'date':
            // in: "auto" tries DateTime; out: PHP date() style? We'll support a small YYYY-MM-DD HH:mm:ss map.
            $in  = $fmt['in']  ?? 'auto';
            $out = $fmt['out'] ?? 'YYYY-MM-DD HH:mm:ss';
            if (!$val) return '';
            try {
                if ($in === 'auto') {
                    $dt = new DateTime($val);
                } else {
                    $dt = DateTime::createFromFormat($in, $val);
                    if (!$dt) $dt = new DateTime($val); // fallback
                }
                // translate simple tokens to PHP format
                $outPhp = strtr($out, [
                    'YYYY'=>'Y','YY'=>'y','MM'=>'m','DD'=>'d',
                    'HH'=>'H','mm'=>'i','ss'=>'s'
                ]);
                return $dt->format($outPhp);
            } catch (Throwable $e) {
                return $val; // fallback raw
            }

        case 'concat':
            $tpl = $fmt['template'] ?? '';
            if ($tpl === '') return '';
            // replace {path} with value from row
            return preg_replace_callback('/\{([^}]+)\}/', function($m) use ($row) {
                $v = getByPath($row, $m[1]);
                return is_scalar($v) ? (string)$v : (is_null($v) ? '' : json_encode($v, JSON_UNESCAPED_SLASHES));
            }, $tpl);

        case 'map':
            $map = $fmt['map'] ?? [];
            $fb  = $fmt['fallback'] ?? '';
            $key = is_scalar($val) ? (string)$val : json_encode($val, JSON_UNESCAPED_SLASHES);
            return array_key_exists($key, $map) ? $map[$key] : $fb;

        default:
            return is_scalar($val) ? $val : (is_null($val) ? '' : json_encode($val, JSON_UNESCAPED_SLASHES));
    }
}

// Header row (visible columns only)
$visibleCols = array_values(array_filter($cols, fn($c) => !isset($c['visible']) || $c['visible']));
fputcsv($out, array_map(fn($c) => $c['header'] ?? ($c['selector'] ?? ''), $visibleCols));

// Data rows
foreach ($rows as $r) {
    $line = [];
    foreach ($visibleCols as $c) {
        $raw = getByPath($r, $c['selector'] ?? '');
        $val = formatValue($raw, $r, $c['fmt'] ?? null);
        // normalize to string
        if (is_bool($val))   $val = $val ? 'true' : 'false';
        if (is_array($val))  $val = json_encode($val, JSON_UNESCAPED_SLASHES);
        if (is_object($val)) $val = json_encode($val, JSON_UNESCAPED_SLASHES);
        $line[] = $val ?? '';
    }
    fputcsv($out, $line,';');
}

fclose($out);
