<?php
/*********************************
 * export_ipath_services_csv.php
 * Streams CSV of ipath services using an internal API call.
 *********************************/

// --- Configuration and Error Setup ---

// Set paths for error logging and application base.
// IMPORTANT: Change this to a writable path on your server!
$errorLogFile = '/tmp/liveconn_csv_export.log';

// Define the base directory (assuming this script is next to 'api' and 'conf.d')
$baseDir = __DIR__ . '/';
$mapFile = $baseDir . '../conf.d/ipath_export_columns.json';
$apiFilePath = $baseDir . '../api/get_ipath_services_page.php';

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', $errorLogFile);
error_reporting(E_ALL);

// Helper for structured logging
function log_message(string $level, string $message, ?array $context = null): void {
    $time = date('Y-m-d H:i:s');
    $logEntry = "[$time] [$level] $message";
    if ($context) {
        $logEntry .= ' | Context: ' . json_encode($context);
    }
    error_log($logEntry);
}

log_message('INFO', 'Starting CSV export process.');

// 1. Set CSV headers immediately
header('Content-Type: text/csv; charset=utf-8');

/*
 * ---- Pass-through filters setup ----
 * Any filters (q, inst, rp, etc.) passed to THIS script via ?q=...&inst[]=...
 * will be visible to get_ipath_services_page.php via $_GET.
 * Here we only normalise offset/limit so the API returns ALL matching rows.
 */

// normalise limit (0 = no limit)
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 0;
if ($limit <= 0) {
    $_GET['limit'] = '0';  // fetch all rows that match current filters
} else {
    $_GET['limit'] = (string)$limit;
}

// normalise offset (export always starts from 0 unless explicitly provided)
$offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
if ($offset < 0) $offset = 0;
$_GET['offset'] = (string)$offset;

log_message('DEBUG', 'GET parameters for internal API call normalised.', [
    'q'      => $_GET['q']      ?? null,
    'inst'   => $_GET['inst']   ?? null,
    'rp'     => $_GET['rp']     ?? null,
    'limit'  => $_GET['limit'],
    'offset' => $_GET['offset'],
]);

// Load column map
if (!file_exists($mapFile)) {
    http_response_code(500);
    log_message('CRITICAL', 'Column map file not found.', ['path' => $mapFile]);
    echo "Configuration error: Column map not found.";
    exit;
}

$mapJson = file_get_contents($mapFile);
$cols = json_decode($mapJson, true);

if (json_last_error() !== JSON_ERROR_NONE || !is_array($cols)) {
    http_response_code(500);
    log_message('CRITICAL', 'Failed to decode column map JSON.', [
        'path'        => $mapFile,
        'json_error'  => json_last_error_msg(),
        'content_start' => substr($mapJson, 0, 50)
    ]);
    echo "Configuration error: Invalid column map JSON.";
    exit;
}

log_message('INFO', 'Column map loaded successfully.', ['column_count' => count($cols)]);

// 2. Fetch data using Output Buffering (Internal Call)
if (!file_exists($apiFilePath)) {
    http_response_code(500);
    log_message('CRITICAL', 'Internal API script not found.', ['path' => $apiFilePath]);
    echo "Internal API script not found.";
    exit;
}

// START OUTPUT BUFFERING to capture the API script's output
ob_start();
try {
    log_message('DEBUG', 'Executing internal API script via require.', ['path' => $apiFilePath]);
    // The API script (get_ipath_services_page.php) now runs using the current $_GET variables
    require $apiFilePath;
    $resp = ob_get_clean();
    log_message('DEBUG', 'Internal API script finished execution.');
} catch (Throwable $e) {
    // If an error occurs, discard the buffer and log the exception.
    if (ob_get_level() > 0) ob_end_clean();
    http_response_code(500);
    log_message('ERROR', 'Error during internal API execution.', [
        'message' => $e->getMessage(),
        'file'    => $e->getFile(),
        'line'    => $e->getLine()
    ]);
    echo "Internal API execution failed.";
    exit;
}

// 3. Process the captured JSON response
if ($resp === '') {
    http_response_code(500);
    log_message('ERROR', 'Internal API call returned empty response.', ['path' => $apiFilePath]);
    echo "Failed to fetch data: Empty response.";
    exit;
}

$json = json_decode($resp, true);

// Check for API failure (invalid JSON or API returning {"ok": false})
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(502);
    log_message('ERROR', 'API returned invalid JSON.', [
        'json_error'     => json_last_error_msg(),
        'response_start' => substr($resp, 0, 256)
    ]);
    echo "Failed to fetch data: Invalid JSON response.";
    exit;
}

if (!($json['ok'] ?? false)) {
    // API executed but failed gracefully (e.g., database error)
    $apiError = $json['error'] ?? 'No specific error message provided.';
    http_response_code(502);
    log_message('ERROR', 'API reported internal failure.', [
        'api_error'      => $apiError,
        'response_start' => substr($resp, 0, 256)
    ]);
    echo "Failed to fetch data from API: " . $apiError;
    exit;
}

$rows = $json['rows'] ?? [];
if (!is_array($rows)) $rows = [];

log_message('INFO', 'Data fetched and decoded successfully.', ['row_count' => count($rows)]);

// Prepare CSV output
$filename = 'ipath_services_' . date('Ymd_His') . '.csv';
header('Content-Disposition: attachment; filename="' . $filename . '"');

// Open output stream
$out = fopen('php://output', 'w');
if ($out === false) {
    http_response_code(500);
    log_message('CRITICAL', 'Failed to open php://output stream.');
    echo "Cannot open output";
    exit;
}

log_message('INFO', 'Output stream opened and headers sent.', ['filename' => $filename]);

// --- Helpers (unchanged) ---

function getByPath(array $arr, ?string $path): mixed {
    if ($path === '' || $path === null) return null;

    $parts = preg_split('/(\.|->)/', $path, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);

    $v = $arr;
    $isJsonDrill = false;

    foreach ($parts as $p) {
        if ($p === '.' || $p === '->') {
            $isJsonDrill = ($p === '->');
            continue;
        }

        if ($isJsonDrill) {
            if (is_string($v)) {
                $jsonDecoded = json_decode($v, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $v = $jsonDecoded;
                } else {
                    return null;
                }
            }
            if (is_array($v) && array_key_exists($p, $v)) {
                $v = $v[$p];
            } else {
                return null;
            }
            $isJsonDrill = false;
        } else {
            if (is_array($v) && array_key_exists($p, $v)) {
                $v = $v[$p];
            } else {
                return null;
            }
        }
    }
    return $v;
}

function formatValue(mixed $val, array $row, ?array $fmt): mixed {
    if (!$fmt || !is_array($fmt) || empty($fmt['type'])) {
        return is_scalar($val) ? $val : (is_null($val) ? '' : json_encode($val, JSON_UNESCAPED_SLASHES));
    }
    switch ($fmt['type']) {
        case 'bool':
            $truthy = $fmt['truthy'] ?? 'Yes';
            $falsy  = $fmt['falsy']  ?? 'No';
            $isTrue = ($val === true) || $val === 1 || $val === '1' || $val === 'true' || $val === 't' || $val === 'Y' || $val === 'yes';
            return $isTrue ? $truthy : $falsy;

        case 'date':
            $in  = $fmt['in']  ?? 'auto';
            $out = $fmt['out'] ?? 'Y-m-d H:i:s';
            if (!$val) return '';
            try {
                if ($in === 'auto') {
                    $dt = new DateTime($val);
                } else {
                    $dt = DateTime::createFromFormat($in, $val);
                    if (!$dt) $dt = new DateTime($val);
                }
                $outPhp = strtr($out, [
                    'YYYY'=>'Y','YY'=>'y','MM'=>'m','DD'=>'d',
                    'HH'=>'H','mm'=>'i','ss'=>'s'
                ]);
                return $dt->format($outPhp);
            } catch (Throwable $e) {
                log_message('WARNING', 'Date formatting failed.', ['value' => $val, 'error' => $e->getMessage()]);
                return $val;
            }

        case 'concat':
            $tpl = $fmt['template'] ?? '';
            if ($tpl === '') return '';
            return preg_replace_callback('/\{([^}]+)\}/', function($m) use ($row) {
                $v = getByPath($row, $m[1]);
                return is_scalar($v) ? (string)$v : (is_null($v) ? '' : json_encode($v, JSON_UNESCAPED_SLASHES));
            }, $tpl);

        case 'map':
            $map = $fmt['map'] ?? [];
            $fb  = $fmt['fallback'] ?? '';
            $key = is_scalar($val) ? (string)$val : json_encode($val, JSON_UNESCAPED_SLASHES);
            return array_key_exists($key, $map) ? $map[$key] : $fb;

        default:
            return is_scalar($val) ? $val : (is_null($val) ? '' : json_encode($val, JSON_UNESCAPED_SLASHES));
    }
}

// --- CSV Generation ---

$visibleCols = array_values(array_filter($cols, fn($c) => !isset($c['visible']) || $c['visible']));
$headers = array_map(fn($c) => $c['header'] ?? ($c['selector'] ?? ''), $visibleCols);
fputcsv($out, $headers);
log_message('INFO', 'CSV header row written.', ['headers' => $headers]);

$rowCount = 0;
foreach ($rows as $r) {
    $line = [];
    foreach ($visibleCols as $c) {
        $selector = $c['selector'] ?? '';
        $raw = getByPath($r, $selector);
        $val = formatValue($raw, $r, $c['fmt'] ?? null);

        if (is_bool($val))   $val = $val ? 'true' : 'false';
        if (is_array($val))  $val = json_encode($val, JSON_UNESCAPED_SLASHES);
        if (is_object($val)) $val = json_encode($val, JSON_UNESCAPED_SLASHES);
        $line[] = $val ?? '';
    }
    fputcsv($out, $line);
    $rowCount++;
}

fclose($out);
log_message('INFO', 'CSV export finished successfully.', ['rows_exported' => $rowCount]);
exit;
