<?php
// api/services_flat_page.php
declare(strict_types=1);
header('Content-Type: application/json');

ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log','/tmp/liveconn.log');
error_reporting(E_ALL);

// Removed the 'hello' and 'exit' lines

require_once '/var/www/html/env_vars.php';
if (!load_env('/etc/netbox/secrets.env'))
{
    http_response_code(500);
    echo json_encode(['ok'=>false, 'message'=>'bad secrets']);
    exit;
}

$DB_HOST = $_SERVER['LIVECONNECTIONS_DATABASE_HOST'];
$DB_PORT = $_SERVER['LIVECONNECTIONS_DATABASE_PORT'];
$DB_NAME = $_SERVER['LIVECONNECTIONS_DATABASE_NAME'];
$DB_USER = $_SERVER['LIVECONNECTIONS_DATABASE_USER'];
$DB_PASS = $_SERVER['LIVECONNECTIONS_DATABASE_PASSWORD'];
$DB_SERVICES_TABLE = $_SERVER['LIVECONNECTIONS_IPATH_SERVICE_TABLE'];
$DB_ROUTING_PLANES_TABLE = $_SERVER['LIVECONNECTIONS_IPATH_ROUTING_PLANES_TABLE'];


function pdo_conn($h,$p,$d,$u,$pw){
    return new PDO("pgsql:host=$h;port=$p;dbname=$d;sslmode=disable", $u, $pw, [
        PDO::ATTR_ERRMODE=>PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_EMULATE_PREPARES=>false
    ]);
}

/**
 * Safely processes a GET parameter that may be a single string, CSV, or an array.
 * Values are trimmed, filtered for ANY/star, and returned as a cleaned array of strings.
 * @param mixed $param The raw value from $_GET.
 * @return array Cleaned list of strings.
 */
function process_list_param(mixed $param): array {
    $list = [];

    // Convert array/string input into a consistent array of strings
    if (is_array($param)) {
        $list = $param;
    } elseif ($param !== null && $param !== '') {
        $list = explode(',', (string)$param);
    }

    // Clean, trim, and make lowercase for consistent comparison
    $list = array_map('trim', $list);
    $list = array_map('strtolower', $list);

    // Remove empty elements and the 'ANY' or '*' filter indicators
    $list = array_filter($list, fn($v) => $v !== '' && !in_array(strtoupper($v), ['ANY', '*'], true));

    return array_values($list);
}


/* ------------ Inputs ------------ */
$offset = max(0, (int)($_GET['offset'] ?? 0));
$rawLimit = (int)($_GET['limit'] ?? 100);
// limit=0 means "all"
$limit = ($rawLimit === 0) ? 0 : min(500, max(1, $rawLimit));

$q = trim((string)($_GET['q'] ?? ''));

// NEW ROBUST FILTERING LOGIC
$instList = process_list_param($_GET['inst'] ?? '');
$rpList   = process_list_param($_GET['rp']   ?? '');

/* ------------ Base SQL ------------ */
$sql = "
SELECT
  d.booking_id,
  d.booking_version,
  LOWER(d.ipath_instance) as ipath_instance,
  d.service_profile,
  d.descriptor_label,
  d.descriptor_description,
  d.from_ipath_logical_source_id,
  d.source_button_name,
  d.from_ip_multicast,
  d.to_ipath_logical_dest_id,
  d.dest_button_name,
  d.profile_type,
  rp.routing_plane_name,
  d.from_sdp_file,
  d.start_fmt,
  d.end_fmt,
  d.created_by,
  d.infinite,

  -- light identifiers for rendering
  d.from_device->>'Device' AS from_device_id,
  d.to_device->>'Device'   AS to_device_id,
  d.from_device,
  d.to_device,

  -- show something pretty if cache has it; otherwise device id (NO lookups here)
  COALESCE(nd_from.netbox_device_name, d.from_device->>'Device') AS from_display_name,
  COALESCE(nd_to.netbox_device_name,   d.to_device->>'Device')   AS to_display_name
FROM {$DB_SERVICES_TABLE} d
JOIN {$DB_ROUTING_PLANES_TABLE} rp ON d.profile_type = rp.profile_tag
LEFT JOIN liveconnections.cache_ipath_netbox_lookup nd_from
  ON nd_from.ipath_device_id = d.from_device->>'Device' AND nd_from.expires > now()
LEFT JOIN liveconnections.cache_ipath_netbox_lookup nd_to
  ON nd_to.ipath_device_id   = d.to_device->>'Device'   AND nd_to.expires > now()
";

//echo($sql);


/* ------------ WHERE clauses ------------ */
$args = [];
$clauses = [];

// inst filter (if provided and not ANY/*)
if (!empty($instList)) {
    $ph = [];
    foreach ($instList as $i => $v) {
        $k = ":inst{$i}";
        $args[$k] = $v;
        $ph[] = $k;
    }
    // FIX: Use LOWER() on the DB column for case-insensitive match against the lowercase parameters
    $clauses[] = "LOWER(d.ipath_instance) IN (".implode(',', $ph).")";
}

// rp filter (if provided and not ANY/*)
if (!empty($rpList)) {
    $ph = [];
    foreach ($rpList as $i => $v) {
        $k = ":rp{$i}";
        $args[$k] = $v;
        $ph[] = $k;
    }
    // NOTE: rpList is lowercased, but routing_plane_name might need to be LOWER()'d too
    // depending on how data is stored. Assuming it's a case-sensitive match for now.
    $clauses[] = "rp.routing_plane_name IN (".implode(',', $ph).")";
}

// free-text search: prefer tsvector column 'full_search' (assumed exists), fallback ILIKEs
if ($q !== '') {
    $clauses[] = "(d.full_search @@ websearch_to_tsquery('simple', :q)
                   OR CAST(d.booking_id AS text) ILIKE :q_like
                   OR d.descriptor_label ILIKE :q_like
                   OR d.source_button_name ILIKE :q_like
                   OR d.dest_button_name ILIKE :q_like
                   OR d.from_ip_multicast ILIKE :q_like)";
    $args[':q'] = $q;
    $args[':q_like'] = '%'.$q.'%';
}

if (!empty($clauses)) {
    $sql .= " WHERE ".implode(' AND ', $clauses);
}

/* ------------ Order / Limit / Offset ------------ */
/* Prefer ordering by a real datetime column if present. Keep your existing column name. */
$sql .= " ORDER BY d.start_fmt DESC";

if ($limit > 0) {
    $sql .= " OFFSET :offset LIMIT :limit";
}

/* ------------ Execute ------------ */
try {
    $pdo = pdo_conn($DB_HOST,$DB_PORT,$DB_NAME,$DB_USER,$DB_PASS);
    $stmt = $pdo->prepare($sql);


    // Bind all named parameters from the filters and search
    foreach ($args as $k => $v) {
        // Use PDO::PARAM_STR for safety, though it often defaults correctly
        $stmt->bindValue($k, $v, PDO::PARAM_STR);
    }

    // Bind limit and offset parameters
    if ($limit > 0) {
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->bindValue(':limit',  $limit,  PDO::PARAM_INT);
    }

    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'ok'          => true,
        'timestamp'   => date('c'),
        'rows'        => $rows,
        'nextOffset'  => ($limit > 0) ? ($offset + count($rows)) : 0,
        'hasMore'     => ($limit > 0) ? (count($rows) === $limit) : false,
        // Debug helpers (optional):
        'debug_sql'   => $sql,
        'debug_args'  => $args
    ], JSON_UNESCAPED_SLASHES);
} catch (Throwable $e) {
    // Log the actual error for debugging
    error_log("DB Error: " . $e->getMessage() . "\nSQL: " . $sql . "\nArgs: " . print_r($args, true));

    http_response_code(500);
    echo json_encode([
        'ok'=>false,
        'error'=>'db_error',
        'message'=>'An internal error occurred while fetching data.' . $e->getMessage()
    ]);
}
