<?php
/*********************************
 * netbox_device_lookup_db.php
 * Lookup a NetBox device by custom field "ipath_device_id"
 * using direct PostgreSQL access.
 *********************************/

ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '../../../env_vars.php';
header('Content-Type: application/json; charset=utf-8');

try {
    if (!load_env('/etc/netbox/secrets.env')) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'bad config']);
        exit;
    }

    $DB_HOST = $_SERVER['NETBOX_DATABASE_HOST'];
    $DB_PORT = $_SERVER['NETBOX_DATABASE_PORT'];
    $DB_NAME = $_SERVER['NETBOX_DATABASE_NAME'];
    $DB_USER = $_SERVER['NETBOX_DATABASE_READONLY_USER'];
    $DB_PASS = $_SERVER['NETBOX_DATABASE_READONLY_PASSWORD'];

    // ipath_id is required
    $ipathId = require_query('ipath_id');

    // --- Start of modifications for optional ipath_instances ---

    $ipathInstancesRaw = trim($_GET['ipath_instances'] ?? '');

    // Normalize and check for 'ANY' or '*'
    $skipInstanceFilter = false;
    if (empty($ipathInstancesRaw) || strtoupper($ipathInstancesRaw) === 'ANY' || $ipathInstancesRaw === '*') {
        $skipInstanceFilter = true;
        $ipathInstances = []; // Empty array if filter is skipped
    } else {
        // Convert comma-separated string to an array of trimmed, non-empty values
        $ipathInstances = array_values(array_filter(array_map('trim', explode(',', $ipathInstancesRaw))));
        // If the resulting array is still empty (e.g., only commas were passed), skip the filter
        if (empty($ipathInstances)) {
            $skipInstanceFilter = true;
        }
    }

    $pdo = new PDO(
        "pgsql:host=$DB_HOST;port=$DB_PORT;dbname=$DB_NAME",
        $DB_USER,
        $DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    // --- Dynamic SQL Construction (Using only positional '?' placeholders) ---

    $positionalParams = [];

    // The base SQL uses a positional placeholder '?' for ipath_device_id
    $sql = <<<SQL
SELECT
    id,
    name,
    serial,
    asset_tag,
    status,
    custom_field_data,
    site_id,
    role_id,
    device_type_id,
    primary_ip4_id,
    primary_ip6_id
FROM dcim_device
WHERE custom_field_data ->> 'ipath_device_id' = ?
SQL;

    // 1. Add ipath_id to the parameters array (corresponds to the first '?' above)
    $positionalParams[] = $ipathId;

    // 2. Dynamically add the ipath_instance filter if required
    if (!$skipInstanceFilter) {
        // Create placeholders for the IN clause (e.g., ?, ?, ?)
        $placeholders = implode(',', array_fill(0, count($ipathInstances), '?'));

        // Append the IN clause to the SQL
        $sql .= " AND custom_field_data ->> 'ipath_instance' IN ({$placeholders})";

        // Append the instance values to the parameters array (in order)
        $positionalParams = array_merge($positionalParams, $ipathInstances);
    }

    // Prepare and execute the statement
    $stmt = $pdo->prepare($sql);

    // Execute using the purely positional array
    $stmt->execute($positionalParams);

    $devices = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (!$devices) {
        echo json_encode(['ok' => true, 'count' => 0, 'devices' => []]);
        exit;
    }

    // Optionally resolve foreign keys for site, role, type, IPs
    $idList = [
        'sites' => [],
        'roles' => [],
        'types' => [],
        'ips'   => [],
    ];

    foreach ($devices as $d) {
        if ($d['site_id'])         $idList['sites'][] = $d['site_id'];
        if ($d['role_id'])  $idList['roles'][] = $d['role_id'];
        if ($d['device_type_id'])  $idList['types'][] = $d['device_type_id'];
        if ($d['primary_ip4_id'])  $idList['ips'][]   = $d['primary_ip4_id'];
        if ($d['primary_ip6_id'])  $idList['ips'][]   = $d['primary_ip6_id'];
    }

    // Helper: fetch lookup tables as simple id->displayName maps
    function lookupMap(PDO $pdo, string $table, array $ids): array {
        if (empty($ids)) return [];
        $ph = implode(',', array_fill(0, count($ids), '?'));

        // choose the display column based on table
        $col = match ($table) {
            'dcim_site'         => 'name',
            'dcim_devicerole'   => 'name',
            'dcim_devicetype'   => 'model',        // ✅ "name" -> "model"
            'ipam_ipaddress'    => 'address',      // IPs
            default             => 'name'
        };

        $sql = "SELECT id, {$col} AS label FROM {$table} WHERE id IN ($ph)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($ids);
        return $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    }

    $mapSite  = lookupMap($pdo, 'dcim_site',        $idList['sites']);
    $mapRole  = lookupMap($pdo, 'dcim_devicerole',  $idList['roles']);
    $mapType  = lookupMap($pdo, 'dcim_devicetype',  $idList['types']);
    $mapIP    = lookupMap($pdo, 'ipam_ipaddress',   $idList['ips']);

    // Format final output
    $baseUrl = rtrim($_SERVER['NETBOX_BASE_URL'] ?? '', '/');

    $out = [];
    foreach ($devices as $d) {
        $out[] = [
            'id'           => (int)$d['id'],
            'name'         => $d['name'],
            'status'       => $d['status'],
            'site'         => $mapSite[$d['site_id']] ?? null,
            'role'         => $mapRole[$d['role_id']] ?? null,
            'device_type'  => $mapType[$d['device_type_id']] ?? null,
            'primary_ip4'  => explode('/', (string)$mapIP[$d['primary_ip4_id']])[0] ?? null,
            'primary_ip6'  => $mapIP[$d['primary_ip6_id']] ?? null,
            'serial'       => $d['serial'],
            'asset_tag'    => $d['asset_tag'],
            'url'          => $baseUrl ? "{$baseUrl}/dcim/devices/{$d['id']}/" : null,
            'custom_fields'=> json_decode($d['custom_field_data'] ?? '{}', true),
        ];
    }

    echo json_encode([
        'ok' => true,
        'timestamp'      => date('c'),
        'count' => count($out),
        'devices' => $out], JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
    
} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: text/plain; charset=utf-8');
    echo "❌ ERROR: " . $e->getMessage() . "\n\n";
    echo $e->getTraceAsString();
}

function require_query(string $name): string {
    $v = trim($_GET[$name] ?? '');
    if ($v === '') {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => "missing ?{$name}= param"]);
        exit;
    }
    return $v;
}