<?php
/*********************************
 * index.php — event-driven UI
 *********************************/
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log','/tmp/liveconn.log');
error_reporting(E_ALL);

require '../env_vars.php';
load_env('/etc/netbox/secrets.env'); // soft load; page still renders w/o it


$SITE_VERSION = "1.1.0";

// ---- Maintenance toggles (env or URL) ----
$DISABLE_LINKS   = (($_SERVER['LIVECONNECTIONS_DISABLE_NETBOX_DEVICE_NAMES']   ?? '') === '1') || isset($_GET['nolinks']);
$NETBOX_BASE_URL = $_SERVER['NETBOX_BASE_URL'] ?? 'https://apishive-ibc-01.dbbroadcast.co.uk/';

// WebSocket endpoints (adjust if needed)
$WS_UPDATE = $_SERVER['LIVE_WS_UPDATE'] ?? 'wss://apishive-xxxx.dbbroadcast.co.uk/nats/services.updates.live_connections.ibc.connection.update';
$WS_REMOVE = $_SERVER['LIVE_WS_REMOVE'] ?? 'wss://apishive-xxxx.dbbroadcast.co.uk/nats/services.updates.live_connections.ibc.connection.remove';

//$WS_STATUS_LDC = $_SERVER['LIVE_WS_STATUS_LDC'] ?? 'ws://127.0.0.1:8080/services.heartbeat.live_connections.ipath_ldc';
//$WS_STATUS_NHN = $_SERVER['LIVE_WS_STATUS_NHN'] ?? 'ws://127.0.0.1:8080/services.heartbeat.live_connections.ipath_nhn';
$WS_STATUS_LDC = $_SERVER['LIVE_WS_STATUS_LDC'] ?? 'wss://apishive-xxxx.dbbroadcast.co.uk/nats/services.heartbeat.live_connections.ibc';
$WS_STATUS_NHN = $_SERVER['LIVE_WS_STATUS_NHN'] ?? 'wss://apishive-xxxx.dbbroadcast.co.uk/nats/services.heartbeat.live_connections.ipath_nhn';
?>
<!doctype html>
<html>
<head>
    <meta charset="utf-8"/>
    <title>HIVE | Live Connections</title>
    <link href="../css/bootstrap.min.css" rel="stylesheet"/>
    <link rel="preload" href="../fonts/roboto/roboto-mono-latin-400-OKRWGZOX.woff2" as="font" type="font/woff2" crossorigin>
    <link rel="stylesheet" href="../fonts/roboto/fonts.css">
    <link rel="icon" href="/favicon.ico">
    <link rel="stylesheet" href="index.css"/>
    <style>
        html, body { height:100%; }
        body{ font-family: system-ui, Segoe UI, Roboto, Helvetica, Arial, sans-serif; margin:16px; }
        .page{ height:100vh; display:flex; flex-direction:column; }
        .topbar{ flex:0 0 auto; margin-bottom:1rem; }
        .content{ flex:1 1 auto; min-height:0; display:flex; }
        .table-scroller{ flex:1 1 auto; min-height:0; overflow:auto; border:1px solid #e5e5e5; border-radius:8px; background:#fff; }

        .dot{ width:12px; height:12px; border-radius:50%; display:inline-block; box-shadow:0 0 0 2px #fff inset,0 0 4px rgba(0,0,0,.2); }
        .dot.red{background:#e74c3c}.dot.amber{background:#f1c40f}.dot.green{background:#2ecc71}

        #connections-table{ width:100%; border-collapse:collapse; }
        #connections-table th, #connections-table td{
            border:1px solid #e5e5e5; padding:3px 6px; font-size:11px; line-height:1.1; vertical-align:middle;
        }
        #connections-table thead th{
            position:sticky; top:0; z-index:2; background:#212529; color:#fff; cursor:pointer; padding-right:20px!important; font-weight: 500;
        }

        #connections-table thead th:nth-child(3),
        #connections-table thead th:nth-child(4),
        #connections-table thead th:nth-child(5),
        #connections-table thead th:nth-child(6),
        #connections-table thead th:nth-child(7){ background:#ffb703; color:#000; }
        #connections-table thead th:nth-child(8),
        #connections-table thead th:nth-child(9),
        #connections-table thead th:nth-child(10){ background:#004466; color:#fff; }

        #connections-table thead th:nth-child(7){ width: 190px}

        /* Device cells & badges */
        td.device-cell{ padding:2px 6px; white-space:nowrap; vertical-align:middle; }
        td.device-cell .device-wrap{ display:flex; align-items:center; gap:6px; width:100%; }

        /* Only anchors should look like links */
        a.device-id{
            color:#0000EE;
            text-decoration:underline;
            cursor:pointer;
        }

        /* Plain text device names should look like text */
        span.device-id{
            color:inherit;
            text-decoration:none;
            cursor:default;
            font-weight:inherit;
        }

        /* When maintenance disables links, you already have this — keep it */
        body.nolinks a.device-id{
            pointer-events:none;
            cursor:default;
            text-decoration:none;
            color:inherit;
        }

        .count-badge{
            margin-left:auto; display:inline-flex; align-items:center; justify-content:center;
            width:1.9em; height:1.9em; border-radius:50%;
            font-size:9px; font-weight:600; line-height:1; text-align:center;
            background: #000; border:1px solid #000; color:#fefe02; box-shadow:inset 0 1px 0 rgba(255,255,255,.15);
            flex-shrink:0;
        }
        .count-badge:hover{ background:#fff; border-color:#000; color:#000; cursor:pointer; }

        /* Logical Sender Id cell: push badge right */
        td:nth-child(4) .device-wrap{ display:flex; align-items:center; justify-content:space-between; gap:6px; width:100%; }

        /* SDP & multicast chips */
        #connections-table td .multicast-badge{
            display:inline-flex!important; align-items:center; justify-content:center;
            padding:0 6px!important; height:16px!important; line-height:16px!important;
            font-size:9px!important; font-weight:400!important; border-radius:10px!important;
            vertical-align:middle!important; margin:0 2px!important; text-transform:uppercase;
            width:80px!important;
        }
        #connections-table td .multicast-badge.red{ background:#c0392b!important; color:#fff!important; }
        #connections-table td .multicast-badge.blue{ background:#2980b9!important; color:#fff!important; }
        #connections-table .sdp-badge.green, #connections-table .sdp-badge.grey{
            display:inline-flex!important; align-items:center; justify-content:center;
            width:44px!important; height:16px!important; border-radius:999px!important;
            font-size:9px!important; font-weight:400!important; text-transform:uppercase;
            letter-spacing:.15px; line-height:1!important; padding:0!important; margin:0 2px!important;
        }
        #connections-table .sdp-badge.green{ background:#198754!important; color:#fff!important; }
        #connections-table .sdp-badge.grey{  background: #acabab !important; color:#000 !important; }
        #connections-table tbody td:nth-child(6),
        #connections-table tbody td:nth-child(7){ text-align:center; vertical-align:middle; }

        #connections-table td:nth-child(12),
        #connections-table th:nth-child(12),
        #connections-table td:nth-child(13),
        #connections-table th:nth-child(13) {
            width: 110px;
            white-space: nowrap;
        }

        .end-infinity{ font-size:1.8em; line-height:.8; display:inline-block; vertical-align:middle; color:#000; }

        .footer{
            position:sticky; bottom:0; z-index:10; background:#fafafa; border-top:1px solid #ddd;
            padding:.4rem .75rem; color:#666; font-size:.85rem; display:flex; justify-content:space-between; align-items:center; height:32px;
        }

        body.nolinks a.device-id{ pointer-events:none; cursor:default; text-decoration:none; color:inherit; }

        .booking-cell { white-space:nowrap; vertical-align:middle; }
        .booking-wrap { display:inline-flex; align-items:center; gap:6px; }
        .action-toggle-btn{
            display:inline-flex;
            align-items:center;
            justify-content:center;
            width:14px; height:14px;
            padding:0; margin:0;
            line-height:1;
            border:0; background:transparent;
            font-size:13px;
        }

        #techInfoModal .modal-dialog{ min-width:900px; }
        /* prettier JSON modal */
        #techInfoModal pre.json-view {
            background:#0b0e14;
            color:#e6e1cf;
            padding:.75rem;
            border-radius:.375rem;
            font-family:ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace;
            font-size:.85rem;
            line-height:1.35;
            overflow:auto;
            white-space:pre;
        }
        #techInfoModal .json-view .key     { color:#89ddff; }
        #techInfoModal .json-view .string  { color:#c3e88d; }
        #techInfoModal .json-view .number  { color:#f78c6c; }
        #techInfoModal .json-view .boolean { color:#ffcb6b; }
        #techInfoModal .json-view .null    { color:#c792ea; }

        .multicast-badge.copied {
            outline: 0;              /* no change, just a hook */
            filter: brightness(1.05);/* subtle flash */
        }

        #ptiModal .modal-dialog{ min-width:900px; }
        .no-data {
            color: red;
            text-align: center;
            font-size: 48px;
        }
        .table-pti tbody tr:nth-child(odd) td,
        .table-pti tbody tr:nth-child(odd) th {
            background-color: #eee;
        }
        .table-pti {
            font-size: 12px;
        }

        .btn-nowrap {
            white-space: nowrap; /* Prevents text from wrapping to the next line */
        }
        .service-status {
            width: 90px;
        }

        /* Custom CSS (e.g., in a style tag or separate CSS file) */
        #fullPageOverlay {
            /* Ensure it covers the whole screen, even if the body is short */
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 1050; /* Above regular content, below modals (z-index 1060+) */
            display: none; /* Hidden by default, JS will toggle using .show */
            opacity: 0.65; /* You can adjust the transparency here if Bootstrap's default isn't enough */
        }

        .overlay-content {
            /* Center content on the screen */
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
            padding: 2rem;
            background-color: rgba(0, 0, 0, 0.5); /* Semi-transparent background for content block */
            border-radius: 0.5rem;
        }

        /* Ensure the overlay is visible when the 'show' class is added by JS */
        .modal-backdrop.show {
            display: block !important;
        }

    </style>
</head>
<body class="<?= $DISABLE_LINKS ? 'nolinks' : '' ?>">
<script>
    // -------- CONFIG you might tweak --------
    const PAGE_SIZE = 150;
    const API = {
        SERVICES_PAGE:  'api/get_ipath_services_page.php',
        SERVICES_ROW:   'api/get_ipath_services_row.php',
        DEVICE_COUNTS: 'api/get_ipath_device_counts.php',
        SENDER_COUNTS:'api/get_ipath_sender_counts.php',
        DISTINCT_VALUES: 'api/get_ipath_distinct_values.php',
        PTI: 'api/get_ipath_receiver_list.php',
        REVERSEPTI: 'api/get_ipath_sender_list.php',
        NETBOX_DEVICE: 'api/get_netbox_device.php',
        BNCS_WORKSTATION: 'api/bncsinfo-proxy.php'
    };
    const WS_UPDATE = <?= json_encode($WS_UPDATE) ?>;
    const WS_REMOVE = <?= json_encode($WS_REMOVE) ?>;
    const WS_STATUS_LDC = <?= json_encode($WS_STATUS_LDC) ?>;
    const WS_STATUS_NHN = <?= json_encode($WS_STATUS_NHN) ?>;
    const BADGE_WS_STATUS_LDC = 'data-ws-ipath-ldc';
    const BADGE_WS_STATUS_NHN = 'data-ws-ipath-nhn';
    const BADGE_STATUS_LDC = 'data-service-ipath-ldc';
    const BADGE_STATUS_NHN = 'data-service-ipath-nhn';

    const NETBOX_BASE_URL = <?= json_encode($NETBOX_BASE_URL) ?>;
    const DISABLE_LINKS = <?= $DISABLE_LINKS ? 'true' : 'false' ?>;

    const RECONNECT_DELAY_MS = 10000; // 10 seconds (configurable delay)
    let timerInterval = null;
    let overlayCountdownTimer = null;

   // const OVERLAY = document.getElementById('fullPageOverlay');
   // const OVERLAY_TIMESTAMP_EL = document.getElementById('overlayTimestamp');
   // const COUNTDOWN_EL = document.getElementById('overlayCountdown');
    </script>

<div class="page">
    <header class="topbar">
        <div class="left-group" style="display:flex;align-items:center;gap:12px 16px;flex-wrap:nowrap;margin-bottom:1.5rem;">
            <a title="Home" href="/"><img style="height:2em" src="/static/logo_netbox_dark_teal.svg" alt="HIVE Logo"></a>
            <img style="height:1.6em" src="logo_nevion.png" alt="Nevion Logo" />
            <h4 class="mb-0 w-100">VideoIPath Live Connections</h4>
            <div class="input-group input-group-sm" style="flex:1;min-width:320px;">
                <span class="input-group-text">Global Search</span>
                <input id="searchInput" type="text" class="form-control" placeholder="type to filter…">
            </div>
            <div class="btn-group btn-group-sm me-2">
                <!-- IPath Instance -->
                <div class="dropdown">
                    <button id="inst-toggle" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown" type="button">
                        IPath Instance — <span id="inst-label">All</span>
                    </button>
                    <div class="dropdown-menu p-2" style="max-height:320px;overflow:auto;min-width:240px;">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="inst-all" checked>
                            <label class="form-check-label" for="inst-all"><strong>All</strong></label>
                        </div>
                        <div id="inst-options" class="mt-1"></div>
                        <button class="btn btn-secondary btn-sm btn-apply" type="button">Close</button>
                    </div>
                </div>
                <!-- Routing Plane -->
                <div class="dropdown">
                    <button id="rp-toggle" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown" type="button">
                        Routing Plane — <span id="rp-label">All</span>
                    </button>
                    <div class="dropdown-menu p-2" style="max-height:320px;overflow:auto;min-width:240px;">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="rp-all" checked>
                            <label class="form-check-label" for="rp-all"><strong>All</strong></label>
                        </div>
                        <div id="rp-options" class="mt-1"></div>
                        <button class="btn btn-secondary btn-sm btn-apply" type="button">Close</button>
                    </div>
                </div>
                <div>
                    <button id="exportCsvBtn" class="btn btn-nowrap btn-primary ms-2" title="Export current view to csv (semi-colon delimited)">
                        💾 Export CSV
                    </button>
                </div>
            </div>
        </div>
    </header>
    <div class="banner-alert" >
    </div>

    <main class="content">
        <div class="table-scroller">
            <table id="connections-table" class="table table-striped table-bordered table-sm table-compact mb-0">
                <thead class="table-dark">
                <tr>
                    <th>Booking</th>
                    <th>Routing Plane</th>
                    <th>From Device</th>
                    <th>Logical Sender Id</th>
                    <th>Source Button Name</th>
                    <th>SDP</th>
                    <th>Multicast</th>
                    <th>To Device</th>
                    <th>Logical Dest Id</th>
                    <th>Dest Button Name</th>
                    <th>Created By</th>
                    <th>Start</th>
                    <th>End</th>
                </tr>
                </thead>
                <tbody id="connections-body"><!-- filled by JS --></tbody>
            </table>

            <div id="sentinel" class="py-2 text-center text-muted">
                <span id="load-more-spinner" style="display:none">Loading…</span>
                <button id="load-more-btn" class="btn btn-sm btn-outline-secondary" style="display:none">Load more</button>
            </div>
        </div>
    </main>

    <!-- SDP Modal -->
    <div class="modal fade" id="sdpModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">SDP File</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <pre id="sdpModalBody" class="mb-0" style="white-space:pre-wrap;"></pre>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" onclick="copySdpFromModal()">Copy</button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Tech Info Modal -->
    <div class="modal fade" id="techInfoModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Tech Info</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-header">
                    <pre id="data-descriptor-label" ></pre>
                </div>
                <div class="modal-body">
                    <h6 class="mb-2">BNCS Information</h6>
                    <pre id="techDetails" class="json-view" style="white-space:pre-wrap;"></pre>
                    <h6 class="mb-2">From Device</h6>
                    <pre id="techFrom" class="json-view" style="white-space:pre-wrap;"></pre>
                    <h6 class="mt-3 mb-2">To Device</h6>
                    <pre id="techTo" class="json-view" style="white-space:pre-wrap;"></pre>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- History Modal -->
    <div class="modal fade" id="historyModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">History</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <h6 class="mb-2">From Device</h6>
                <pre id="descriptorDescription" class="json-view" style="white-space:pre-wrap;"></pre>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- PTI Modal -->
    <div class="modal fade" id="ptiModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="ptiBanner">Destination List</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="pti-table" id="ptiPayload"></div>
                </div>

                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <footer class="footer px-3">
        <div id="record-count" class="text-muted small">0 records</div>
        <div class="text-center flex-grow-1 small fw-semibold text-secondary" id="data-site-version">Powered by HIVE</div>
        <div class="status-dots" style="display:flex;gap:8px;align-items:center">
            <div>
                <h6 class="badge rounded-pill" id="data-ws-ipath-ldc">LDC</h6>
            </div>
            <div>
                <h6 class="badge service-status text-bg-danger" id="data-service-ipath-ldc">FAILED</h6>
            </div>
            <div>
                <h6 class="badge rounded-pill" id="data-ws-ipath-nhn">NHN</h6>
            </div>
            <div>
                <h6 class="badge service-status text-bg-warning" id="data-service-ipath-nhn">PENDING</h6>
            </div>
            <div class="text-muted small" id="ws-timestamp">Last update: ----</div>
            <span id="dot-update" class="dot red" title="Service Changes"></span>
            <span id="dot-remove"  class="dot red" title="Service Removals"></span>
        </div>
    </footer>
</div>

<div class="modal-backdrop fade" id="fullPageOverlay">
    <div id="overlayContent" class="overlay-content">
        <h1 class="text-red mb-2" style="color:red; font-size=120;">CRITICAL SERVICE ALERT</h1>
        <h2 class="text-red mb-4">IPath disconnect or failover event</h2>

        <p class="text-white lead mb-2">Last Status Change: <span id="overlayTimestamp">...</span></p>

        <p class="text-white lead">Reconnection Attempt In: <span id="overlayCountdown">...</span> seconds</p>
    </div>
</div>

<script src="../static/bootstrap-5.3.3/bootstrap.bundle.min.js"></script>
<script>
    // set up our lookup for glyphs - could move to SQL
    window.RoutingPlaneGlyphLookup = new Map([
        ["Video", "📺"],
        ["Audio", "🎤"],
        ["Anc", "␐"],
        ["JPEGXS", "🗜"],
    ]);
    $siteVersion = 'Site Version <?php echo $SITE_VERSION ?>';

    let __io = null;
    let __stopUpdate = null;
    let __stopRemove = null;

    const pendingDevIds = new Set();
    let devRefreshTimer = null;

    const pendingSenderIds = new Set();
    let senderRefreshTimer = null;

    function queueSenderCountRefresh(ids) {
        (ids || []).forEach(id => {
            const v = (id || '').trim();
            if (v) pendingSenderIds.add(v);
        });
        if (senderRefreshTimer) return;

        senderRefreshTimer = setTimeout(async () => {
            const batch = [...pendingSenderIds];
            pendingSenderIds.clear();
            senderRefreshTimer = null;
            if (batch.length) await refreshSenderCounts(batch);
        }, 50);
    }

    // one cache, one resolver, one shape: { id:Number, name:String } or null
    const nbDeviceCache = new Map(); // ipathId -> { t, v }

    async function getNetboxDevice(ipathId, ttlMs = 5 * 60 * 1000) {
        const idStr = String(ipathId || '').trim();
        if (!idStr) return null;

        // Honour maintenance mode: short-circuit (no lookups, no linking)
        if (DISABLE_LINKS) return null;

        const hit = nbDeviceCache.get(idStr);
        if (hit && (Date.now() - hit.t) < ttlMs) return hit.v;

        try {
            // Use the same-origin API constant for consistency
            const url = new URL(API.NETBOX_DEVICE, window.location.href);
            url.searchParams.set('ipath_id', idStr);
            url.searchParams.set('ipath_instances', 'ANY');

            const res = await fetch(url.toString(), { cache: 'no-store' });
            if (!res.ok) throw new Error('HTTP ' + res.status);

            const json = await res.json();
            const dev = json?.devices?.[0] || null;

            // Normalize shape; make sure id is a number (NetBox URLs tolerate numeric ids)
            const value = dev ? { id: Number(dev.id) || 0, name: String(dev.name || idStr) } : null;

            nbDeviceCache.set(idStr, { t: Date.now(), v: value });
            return value;
        } catch {
            const value = null;
            nbDeviceCache.set(idStr, { t: Date.now(), v: value });
            return value;
        }
    }


    function mountInfiniteScroll() {
        const sentinel = document.getElementById('sentinel');
        if (__io) { try { __io.disconnect(); } catch {} }
        __io = new IntersectionObserver((entries) => {
            if (entries[0].isIntersecting && !loading && hasMore) fetchPage();
        }, { root: document.querySelector('.table-scroller'), threshold: 0, rootMargin: '0px 0px 200px 0px' });

        sentinel && __io.observe(sentinel);
    }

    function mountSockets() {
        const dotUpdate = document.getElementById('dot-update');
        const dotRemove = document.getElementById('dot-remove');

        if (__stopUpdate) { try { __stopUpdate(); } catch {} }
        if (__stopRemove) { try { __stopRemove(); } catch {} }

        __stopUpdate = startWS(WS_UPDATE, dotUpdate, onUpdateArrived);
        __stopRemove = startWS(WS_REMOVE, dotRemove, onRemoveArrived);


    }

    window.addEventListener('beforeunload', () => {
        try { __io && __io.disconnect(); } catch {}
        try { __stopUpdate && __stopUpdate(); } catch {}
        try { __stopRemove && __stopRemove(); } catch {}
    });


    /* ---------- Tiny helpers ---------- */
    // html-escape a string (for safe insertion into HTML)
    const esc = s => s==null?'':String(s).replace(/[&<>"']/g, m=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;'}[m]));

    // base64 encode (UTF-8 safe)
    const b64encUtf8 = str => btoa(unescape(encodeURIComponent(str)));

    // base64 decode (UTF-8 safe)
    const b64decUtf8 = b64 => { try { return decodeURIComponent(escape(atob(b64))); } catch { return atob(b64); } };

    // determine if a value is "truthy" in the loose human sense (strings like "true", "1")
    const isTrueish = v => v===true || v===1 || v==='1' || v==='t' || v==='true';

    // set dot indicator element to red/amber/green
    function setDot(el, state){ if(!el) return; el.classList.remove('red','amber','green'); el.classList.add(state); }

    // update a timestamp on a page element called ws-timestamp
    function updateWsTimestamp(){
        const el = document.getElementById('ws-timestamp'); if(!el) return;
        const now = new Date(); el.textContent = `Last update: ${now.toTimeString().slice(0,8)}`;
    }

    function queueDeviceCountRefresh(ids) {
        (ids || []).forEach(id => {
            const v = (id || '').trim();
            if (v) pendingDevIds.add(v);
        });
        if (devRefreshTimer) return;

        // batch within ~1 tick
        devRefreshTimer = setTimeout(async () => {
            const batch = [...pendingDevIds];
            pendingDevIds.clear();
            devRefreshTimer = null;
            if (batch.length) await refreshDeviceCounts(batch);
        }, 50);
    }

    // parse JSON or double-parsed JSON, otherwise return raw text
    function parseOnceOrTwice(txt){
        try { return JSON.parse(txt); } catch {}
        try { return JSON.parse(JSON.parse(txt)); } catch {}
        return { _raw: txt };
    }

    // recursively sort object keys deeply
    function sortDeep(v){
        if (Array.isArray(v)) return v.map(sortDeep);
        if (v && typeof v === 'object') {
            return Object.keys(v).sort().reduce((a,k)=>{ a[k]=sortDeep(v[k]); return a; },{});
        }
        return v;
    }


    // normalize arbitrary input (object / JSON text / base64 blob) into a JS object
    function normalizeJsonInput(input){
        if (input == null) return {};
        if (typeof input === 'object') return input;

        // input is text (possibly quoted JSON)
        let txt = String(input).trim();

        // If it looks like a base64-encoded blob from our data-attrs, decode first
        if (/^[A-Za-z0-9+/=]+$/.test(txt) && txt.length % 4 === 0) {
            try { txt = b64decUtf8(txt); } catch {}
        }

        // Try once
        try { return JSON.parse(txt); } catch {}

        // If it was a JSON string containing JSON, unquote-and-parse again
        if (txt.startsWith('"') && txt.endsWith('"')) {
            try { return JSON.parse(JSON.parse(txt)); } catch {}
        }

        // Last resort: show raw text
        return { _raw: txt };
    }

    // pretty-print obj into JSON (with stable key sorting)
    function pretty(obj){
        // stable key order for nicer diffs
        const sortDeep = v => Array.isArray(v)
            ? v.map(sortDeep)
            : (v && typeof v === 'object')
                ? Object.keys(v).sort().reduce((a,k)=>{ a[k]=sortDeep(v[k]); return a; },{})
                : v;
        return JSON.stringify(sortDeep(obj), null, 2);
    }

    // deep sort all keys (deterministic ordering)
    function sortObjectDeep(value){
        if (Array.isArray(value)) return value.map(sortObjectDeep);
        if (value && typeof value === 'object') {
            return Object.keys(value).sort().reduce((acc,k)=>{ acc[k]=sortObjectDeep(value[k]); return acc; },{});
        }
        return value;
    }

    // colorize JSON for HTML syntax highlighting
    function syntaxHighlightJSON(text){
        const esc = text.replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
        return esc.replace(/("(\\u[\da-fA-F]{4}|\\[^u]|[^\\"])*"(\s*:)?|\b(true|false|null)\b|-?(?:\d+(?:\.\d+)?)\b)/g, (m)=>{
            let cls='number';
            if (m[0] === '"') cls = /:$/.test(m) ? 'key' : 'string';
            else if (/true|false/.test(m)) cls='boolean';
            else if (/null/.test(m)) cls='null';
            return '<span class="'+cls+'">'+m+'</span>';
        });
    }

    // render object as sorted JSON, highlighted like jq output
    function renderJQLike(obj){
        const sorted = sortObjectDeep(obj);
        const pretty = JSON.stringify(sorted, null, 2);
        return syntaxHighlightJSON(pretty);
    }

    function copyToClipboard(text, el) {
        const showTick = () => {
            if (!el) return;
            const old = el.textContent;
            el.dataset._oldText = old;
            el.textContent = '✔';
            el.title = 'Copied!';
            el.classList.add('copied'); // optional css hook
            setTimeout(() => {
                el.textContent = el.dataset._oldText || old;
                el.title = 'Copy ' + (el.dataset._oldText || old);
                el.classList.remove('copied');
            }, 666);
        };

        // Modern API
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(String(text))
                .then(showTick)
                .catch(() => { legacyCopy(String(text)); showTick(); });
            return;
        }
        // Fallback
        legacyCopy(String(text)); showTick();

        function legacyCopy(t) {
            const ta = document.createElement('textarea');
            ta.value = t;
            ta.style.position = 'fixed';
            ta.style.opacity = '0';
            document.body.appendChild(ta);
            ta.focus();
            ta.select();
            try { document.execCommand('copy'); } catch {}
            document.body.removeChild(ta);
        }
    }

    // Cache to avoid hammering the proxy
    const _wsDataCache = new Map(); // id -> { short_name, name, ... } or null

    // Resolve a BNCS workstation record by integer id via same-origin proxy.
    // Returns the underlying data object (or null on failure).
    async function resolveWorkstationData(w) {
        const id = Number.parseInt(String(w), 10);
        if (!Number.isFinite(id)) return null;

        if (_wsDataCache.has(id)) return _wsDataCache.get(id);

        try {
            const url = new URL(API.BNCS_WORKSTATION, window.location.href);
            url.searchParams.set('id', String(id));
            const res = await fetch(url.toString(), { cache: 'no-store' });
            if (!res.ok) throw new Error('HTTP ' + res.status);

            const j = await res.json();
            // your proxy returns { ok, data, ... }
            const data = j?.data || j || null;
            _wsDataCache.set(id, data);
            return data;
        } catch (err) {
            console.warn('[workstation lookup] failed', err);
            _wsDataCache.set(id, null);
            return null;
        }
    }


    // Finds <span class="device-id" data-ipath-id="..."> and hydrates to <a> (or renames span)
    // This is the ONLY place we create anchors for device links.
    async function resolveDeviceBadges(root = document) {
        const nodes = Array.from(root.querySelectorAll('span.device-id[data-ipath-id]'));
        if (!nodes.length) return;

        // Deduplicate by ipathId → list of elements
        const byId = new Map();
        for (const el of nodes) {
            const ipathId = (el.getAttribute('data-ipath-id') || '').trim();
            if (!ipathId) continue;
            if (!byId.has(ipathId)) byId.set(ipathId, []);
            byId.get(ipathId).push(el);
        }

        for (const [ipathId, els] of byId.entries()) {
            const info = await getNetboxDevice(ipathId); // null if disabled or not found

            for (const el of els) {
                const title = el.getAttribute('title') || `ipath ${ipathId}`;
                const label = info?.name || el.textContent || ipathId || '—';

                if (info?.id && !DISABLE_LINKS) {
                    const a = document.createElement('a');
                    a.className = 'device-id';
                    a.href = new URL(`dcim/devices/${info.id}/`, NETBOX_BASE_URL).toString();
                    a.target = '_blank';
                    a.rel = 'noopener';
                    a.title = title;
                    a.textContent = label;
                    el.replaceWith(a);
                } else {
                    // Links disabled or no NB device found → just show a clean name as plain text
                    el.textContent = label;
                    el.title = title;
                }
            }
        }
    }


    // parse various date-ish inputs (including 'YYYY-MM-DD HH:MM') into a Date or null
    function parseEndToDate(raw){
        if (!raw) return null;
        let d = new Date(raw);
        if (!isNaN(d)) return d;
        d = new Date(String(raw).replace(' ', 'T'));
        return isNaN(d) ? null : d;
    }

    // map an end-time to a Bootstrap table severity class (danger/warning/none)
    function timeSeverityClass(endRaw){
        const dt = parseEndToDate(endRaw);
        if (!dt) return '';
        const diff = dt.getTime() - Date.now();
        if (diff <= 0) return 'table-danger';
        if (diff <= 5 * 60 * 1000) return 'table-danger';
        if (diff <= 30 * 60 * 1000) return 'table-warning';
        return '';
    }

    // format a millisecond countdown as compact human-readable text
    function formatCountdown(ms){
        if (ms <= 0) return 'Expired';
        const sec = Math.floor(ms/1000);
        const d = Math.floor(sec / 86400);
        const h = Math.floor((sec % 86400) / 3600);
        const m = Math.floor((sec % 3600) / 60);
        const s = sec % 60;

        if (d > 0) return `${d}d ${h}h ${m}m`;
        if (h > 0) return `${h}h ${m}m ${s}s`;
        if (m > 0) return `${m}m ${s}s`;
        return `${s}s`;
    }

    let _countdownTimer = null;

    // refresh all .end-countdown elements and update row severity classes
    function updateAllCountdowns(){
        const now = Date.now();
        document.querySelectorAll('.end-countdown').forEach(el=>{
            const raw = el.getAttribute('data-end-raw') || '';
            const dt  = parseEndToDate(raw);
            if (!dt){ el.textContent = raw || '—'; return; }

            const diff = dt.getTime() - now;
            el.textContent = formatCountdown(diff);

            const tr = el.closest('tr');
            if (tr){
                tr.classList.remove('table-danger','table-warning');
                tr.classList.add(timeSeverityClass(raw)); // same rule as initial render
            }
        });
    }

    // start (or restart) the global countdown interval and do an immediate tick
    function startCountdownTicker(intervalMs = 1000){
        if (_countdownTimer) clearInterval(_countdownTimer);
        updateAllCountdowns(); // prime immediately
        _countdownTimer = setInterval(updateAllCountdowns, intervalMs);
    }



    /* ---------- Facets ---------- */
    window.filterState = { inst: [], rp: [], q: '' };
    // set a facet label to 'All' / specific value / 'Multi' based on selection list
    function setFacetLabel(el, list){ if(!el) return; el.textContent = !list?.length ? 'All' : (list.length===1 ? list[0] : 'Multi'); }

    // Coerce any value into a simple array for facet building
    function asArray(v) {
        if (Array.isArray(v)) return v;
        if (v == null) return [];
        if (typeof v === 'object') {
            // if API ever returns { "0": "ibc", "1": "prod" } etc.
            return Object.values(v);
        }
        const s = String(v).trim();
        if (!s) return [];
        return s.split(',').map(x => x.trim()).filter(Boolean);
    }

    /* ---------- Facets ---------- */
    window.filterState = { inst: [], rp: [], q: '' };
    // set a facet label to 'All' / specific value / 'Multi' based on selection list
    function setFacetLabel(el, list){ if(!el) return; el.textContent = !list?.length ? 'All' : (list.length===1 ? list[0] : 'Multi'); }

    // load distinct filter values, rebuild facet checklists, persist state, and trigger reload
    async function loadDistinctsAndFacets()
    {
        try
        {
            const r = await fetch(API.DISTINCT_VALUES, {cache: 'no-store'});
            if (!r.ok) throw 0;
            const d = await r.json();
            if (!d.ok) throw 0;
            const saved = JSON.parse(localStorage.getItem('liveFilters') || '{}');
            filterState.inst = Array.isArray(saved.inst) ? saved.inst : [];
            filterState.rp = Array.isArray(saved.rp) ? saved.rp : [];

            // build a checkbox list for a facet and wire up persistence + reload
            const build = (values, wrapId, allId, labelId, key) => {
                const wrap = document.getElementById(wrapId);
                const all = document.getElementById(allId);
                const label = document.getElementById(labelId);

                const list = asArray(values);   // ← always an array now

                wrap.innerHTML = '';
                list.forEach(v => {
                    const id = `${wrapId}-${Math.random().toString(36).slice(2)}`;
                    const row = document.createElement('div');
                    row.className = 'form-check';
                    const inp = document.createElement('input');
                    inp.type = 'checkbox';
                    inp.className = 'form-check-input';
                    inp.id = id;
                    inp.value = String(v);
                    inp.checked = filterState[key].includes(v);
                    const lab = document.createElement('label');
                    lab.className = 'form-check-label';
                    lab.htmlFor = id;
                    lab.textContent = String(v);
                    row.appendChild(inp);
                    row.appendChild(lab);
                    wrap.appendChild(row);
                });

                const apply = () => {
                    const sel = Array.from(wrap.querySelectorAll('input:checked')).map(i => i.value);
                    filterState[key] = sel;
                    localStorage.setItem('liveFilters', JSON.stringify({inst: filterState.inst, rp: filterState.rp}));
                    setFacetLabel(label, sel);
                    // reload fresh
                    resetAndLoad();
                };
                wrap.addEventListener('change', apply);
                all.addEventListener('change', e => {
                    if (e.target.checked) {
                        wrap.querySelectorAll('input').forEach(i => i.checked = false);
                        apply();
                    }
                });
                setFacetLabel(label, filterState[key]);
            };

            build(d.ipath_instance || [], 'inst-options', 'inst-all', 'inst-label', 'inst');
            build(d.routing_plane || [], 'rp-options', 'rp-all', 'rp-label', 'rp');
        }
        catch (e)
        {
            console.warn('[facets] failed', e);
        }

    }


    /* ---------- Rendering ---------- */

    // render an SDP badge, embedding SDP in base64 on the element for later viewing
    function sdpBadgeHtml(sdp){
        const val = (sdp ?? '').trim();
        if (!val) return '<span class="badge sdp-badge grey" title="No SDP">SDP</span>';
        const b64 = b64encUtf8(val);
        return `<span class="badge sdp-badge green" role="button" tabindex="0" data-sdp="${b64}" title="View SDP">SDP</span>`;
    }

    // render up to two multicast address badges (L/R), each clickable to copy
    function multicastBadgesHtml(v){
        if (v == null) return '';
        const parts = String(v).split('|').map(s=>s.trim()).filter(Boolean).slice(0,2);
        if (!parts.length) return '';
        const halves = parts.map((ip,i)=>{
            const justify = i===0?'justify-content-start':'justify-content-end';
            const color   = i===0?'red':'blue';
            const safe    = ip.replace(/'/g,'&#39;');
            return `<div class="d-flex ${justify}" style="flex:1">
      <span class="badge multicast-badge ${color}" style="cursor:pointer" title="Copy ${safe}"
            onclick="copyToClipboard('${safe}', this)">${ip}</span>
    </div>`;
        });
        if (halves.length===1) halves.push('<div class="d-flex justify-content-end" style="flex:1"></div>');
        return `<div class="d-flex justify-content-between align-items-center" style="width:100%;gap:2px">${halves.join('')}</div>`;
    }

    // pick a friendly display name for a row's 'from' or 'to' side, with fallback
    function getDisplayName(row, side){
        const disp = side==='from' ? (row.from_display_name||'') : (row.to_display_name||'');
        return (disp||'').trim() || '—';
    }

    // render a placeholder <span> for a device (to be hydrated into a link later)
    function deviceLinkOrSpan(row, side){
        const ipathId = side === 'from' ? (row.from_device_id || '') : (row.to_device_id || '');
        const title   = `ipath ${esc(ipathId)}`;

        // Initial text (fast): whatever display name you already have, else the id
        const initialText = esc((side === 'from' ? row.from_display_name : row.to_display_name) || ipathId || '—');

        // Always render a <span> first; we'll hydrate to <a> if allowed + resolvable
        return `<span class="device-id" data-ipath-id="${esc(ipathId)}" data-side="${side}" title="${title}">${initialText}</span>`;
    }

    const nbResolveCache = new Map(); // ipathId -> { id, name } or null


    // render a full live-connections table row with data-attrs, badges, and countdown
    function renderRowHtml(row){
        const bId = esc(row.booking_id);
        const bVer = esc(row.booking_version);
        const id = `r-${bId}-${bVer}`;
        const isInf = isTrueish(row.infinite);
        const severity = isInf ? '' : timeSeverityClass(row.end_fmt);
        const senderId = esc(row.from_ipath_logical_source_id ||'');
        const sourceBtnName = esc(row.source_button_name || 'tt');
        const ipathInstance = row.ipath_instance;
        const fromId = esc(row.from_device_id ||'');
        const toId   = esc(row.to_device_id ||'');
        const fromDevice = esc(row.from_device ||'');
        const toDevice   = esc(row.to_device ||'');
        const dLabel = esc(row.descriptor_label);
        // Prefer any of these keys; supports both DB render and row-API
        const fromPayload = row.from_device ?? row.from_device_data ?? row.from_device_json ?? null;
        const toPayload   = row.to_device   ?? row.to_device_data   ?? row.to_device_json   ?? null;
        const serviceDescriptorLabel  = row.descriptor_label  ?? null;
        const serviceDescriptorDesc  = row.descriptor_description  ?? null;

        // If it’s already a string, use it; else stringify the object
        const fromRaw = (typeof fromPayload === 'string') ? fromPayload : JSON.stringify(fromPayload || {});
        const toRaw   = (typeof toPayload   === 'string') ? toPayload   : JSON.stringify(toPayload   || {});
        const detailsRaw   = (typeof serviceDescriptorDesc    === 'string') ? serviceDescriptorDesc   : JSON.stringify(serviceDescriptorDesc   || {});

        // Base64 for data-attrs
        const fromB64 = b64encUtf8(fromRaw);
        const toB64   = b64encUtf8(toRaw);
        const detailsB64   = b64encUtf8(detailsRaw);

        const routingPlaneGlyph = RoutingPlaneGlyphLookup.get(row.routing_plane_name) ?? '❓';

        return `
<tr id="${id}" class="${severity}" data-ipath-instance="${ipathInstance}" data-from-id="${fromId}" data-to-id="${toId}" data-from-b64="${fromB64}" data-to-b64="${toB64}" data-sender-id="${senderId}" data-sce-btn="${sourceBtnName}" data-descriptor-label="${serviceDescriptorLabel}" data-desc-b64="${detailsB64}">
<td class="booking-cell align-middle" title="Route: ${esc(row.descriptor_label ?? '')}">
      <div class="booking-wrap">
        <button
          class="action-toggle-btn popover-launch"
          type="button"
          aria-label="More actions"
        >&#8801;</button>
        ${esc(row.booking_id)}-${esc(row.booking_version)}
      </div>
    </td>
  <td>
     <div>
         <span>${routingPlaneGlyph}</span>
         <span>${esc(row.routing_plane_name)}</span>
     </div>
</td>
  <td class="device-cell">
    <div class="device-wrap">
      ${deviceLinkOrSpan(row,'from')}
      <span class="count-badge" data-badge="from" style="display:none">0</span>
    </div>
  </td>

  <td>
    <div class="device-wrap">
      <span>${esc(row.from_ipath_logical_source_id)}</span>
      <span class="count-badge" data-badge="sender" data-pop="pti" style="display:none">0</span>
    </div>
  </td>

  <td>${esc(row.source_button_name)}</td>
  <td>${sdpBadgeHtml(row.from_sdp_file)}</td>
  <td>${multicastBadgesHtml(row.from_ip_multicast)}</td>

  <td class="device-cell">
    <div class="device-wrap">
      ${deviceLinkOrSpan(row,'to')}
      <span class="count-badge" data-badge="to" style="display:none">0</span>
    </div>
  </td>

  <td>${esc(row.to_ipath_logical_dest_id)}</td>
  <td>${esc(row.dest_button_name)}</td>
  <td>${esc(row.created_by)}</td>
  <td>${esc(row.start_fmt)}</td>
  <td>
    ${
            isInf
                ? '<span class="end-infinity" title="Infinite">&#8734;</span>'
                : `<span class="end-countdown"
                 data-end-raw="${esc(row.end_fmt || '')}"
                 title="${esc(row.end_fmt || '')}"></span>`
        }
  </td>
</tr>`;
    }

    // render a compact PTI-only row (subset of columns) for the PTI view
    function renderPTIRowHtml(row){
        const bId = esc(row.booking_id);
        const bVer = esc(row.booking_version);
        const id = `r-${bId}-${bVer}`;

        return `
<tr id="${id}" >
<td class="booking-cell align-middle" title="Route: ${esc(row.descriptor_label ?? '')}">
      <div class="booking-wrap">
        <button
          class="action-toggle-btn popover-launch"
          type="button"
          aria-label="More actions"
        >&#8801;</button>
        ${esc(row.booking_id)}-${esc(row.booking_version)}
      </div>
    </td>
  <td class="device-cell">
    <div class="device-wrap">
      ${deviceLinkOrSpan(row,'from')}
      <span class="count-badge" data-badge="from" style="display:none">0</span>
    </div>
  </td>
  <td>${esc(row.to_ipath_logical_dest_id)}</td>
  <td>${esc(row.dest_button_name)}</td>
</tr>`;
    }

    /* ---------- SDP modal open ---------- */
    // open the SDP viewer modal when an .sdp-badge.green is clicked
    document.addEventListener('click', e=>{
        const el = e.target.closest('.sdp-badge.green');
        if (!el) return;
        const b64 = el.getAttribute('data-sdp'); if (!b64) return;
        const text = b64decUtf8(b64);
        document.getElementById('sdpModalBody').textContent = text;
        bootstrap.Modal.getOrCreateInstance(document.getElementById('sdpModal')).show();
    });

    // fetch and show a bookings table in a modal when a .count-badge is clicked (with live counts)
    document.addEventListener('click', async (e) => {
        const badge = e.target.closest('.count-badge');
        if (!badge) return;

        // Which row / ids?
        const tr = badge.closest('tr');
        const sender = tr?.getAttribute('data-sender-id') || '';
        const to = tr?.getAttribute('data-to-id') || '';
        const sceBtnName = tr?.getAttribute('data-sce-btn') || '';

        // Decide what the badge represents (sender vs from/to)
        const kind = badge.dataset.badge || ''; // "sender" | "from" | "to"

        // Declare variables
        let url;
        let payload;
        let rows;
        let headerText;
        let html;

        try {
            // --- 1. Determine the URL for the main 'bookings' payload ---
            if (kind === 'sender' && sender) {
                // Use PTI for sender
                url = new URL(API.PTI, window.location.href);
                url.searchParams.set('sender', sender);
            } else if (kind === 'to' && to) {
                // Use REVERSEPTI for 'to' bookings
                url = new URL(API.REVERSEPTI, window.location.href);
                url.searchParams.set('device_id', to);
            } else {
                console.warn('[badge] no suitable key to query', { kind, sender, to });
                return;
            }

            // --- 2. Await the Main Booking Data Fetch (For both 'sender' and 'to') ---
            const res = await fetch(url.toString(), { cache: 'no-store' });

            if (!res.ok) {
                throw new Error(`HTTP Error from main API (${url.pathname}): ${res.status}`);
            }

            // Await response parsing as JSON (for bookings)
            payload = await res.json();
            console.log(payload);

            // --- 3. Process the Data and Build HTML ---

            if (kind === "sender") {
                // API.PTI case
                headerText = `Destination List for ${sender} (${sceBtnName})`;
                rows = (payload.bookings || []).map(r => `
                <tr>
                    <td>${esc(r.booking_id)}-${esc(r.booking_version)}</td>
                    <td>${esc(r.to_ipath_logical_dest_id || '')}</td>
                    <td>${esc(r.dest_button_name || '')}</td>
                    <td>${esc(r.created_by || '')}</td>
                    <td>${esc(r.start_fmt || '')}</td>
                    <td>
                        ${
                    r.infinite
                        ? '<span class="end-infinity" title="Infinite">&#8734;</span>'
                        : `<span class="end-countdown"
                                data-end-raw="${esc(r.end_fmt || '')}"
                                title="${esc(r.end_fmt || '')}"></span>`
                }
                    </td>
                </tr>
            `).join('') || '<tr><td colspan="6" class="text-muted">No rows</td></tr>';

                html = `
                <table class="table table-sm table-bordered table-pti mb-0">
                    <thead>
                        <tr>
                            <th>Booking</th>
                            <th>Logical Dest Id</th>
                            <th>Name</th>
                            <th>Created</th>
                            <th>Start</th>
                            <th>End</th>
                        </tr>
                    </thead>
                    <tbody>${rows}</tbody>
                </table>
            `;

            } else if (kind == "to") {
                // API.REVERSEPTI + API.NETBOX_DEVICE case

                // --- AWAIT the SECOND API Call (NETBOX_DEVICE) for header text ---
                const url1 = new URL(API.NETBOX_DEVICE, window.location.href);
                url1.searchParams.set('ipath_id', to);

                if (DISABLE_LINKS)
                {
                    headerText = `All Source List for ${to}`;
                }
                else {
                    const res1 = await fetch(url1.toString(), {cache: 'no-store'});
                    if (!res1.ok) {
                        // Handle failure of the secondary call gracefully in the header
                        console.warn(`[badge click] NETBOX_DEVICE fetch failed with status: ${res1.status}`);
                        headerText = `Source List for device: ${to} (Error loading device info)`;
                    } else {
                        const devicePayload = await res1.json();
                        const deviceName = devicePayload?.devices?.[0]?.name ?? 'Unknown Device';
                        headerText = `Source List for ${deviceName} (${to})`;
                    }
                }

                // Build rows using the main 'payload' (from API.REVERSEPTI)
                rows = (payload.bookings || []).map(r => `
                <tr>
                    <td>${esc(r.booking_id)}-${esc(r.booking_version)}</td>
                    <td>${esc(r.from_ipath_logical_source_id || '')}</td>
                    <td>${esc(r.to_ipath_logical_dest_id || '')}</td>
                    <td>${esc(r.source_button_name || '')}</td>
                    <td>${esc(r.created_by || '')}</td>
                    <td>${esc(r.start_fmt || '')}</td>
                    <td>
                        ${
                    r.infinite
                        ? '<span class="end-infinity" title="Infinite">&#8734;</span>'
                        : `<span class="end-countdown"
                                data-end-raw="${esc(r.end_fmt || '')}"
                                title="${esc(r.end_fmt || '')}"></span>`
                }
                    </td>
                </tr>
            `).join('') || '<tr><td colspan="7" class="text-muted">No rows</td></tr>';

                html = `
                <table class="table table-sm table-bordered table-pti mb-0">
                    <thead>
                        <tr>
                            <th>Booking</th>
                            <th>Logical Source Id</th>
                            <th>Logical Dest Id</th>
                            <th>Name</th>
                            <th>Created</th>
                            <th>Start</th>
                            <th>End</th>
                        </tr>
                    </thead>
                    <tbody>${rows}</tbody>
                </table>
            `;
            }

            // Ensure content was generated before rendering
            if (!html) return;

            // --- 4. Render and Show Modal (Runs ONLY after all AWAITS are resolved) ---
            console.log("Rendering modal with fetched data...");
            document.getElementById('ptiPayload').innerHTML = html;
            document.getElementById('ptiBanner').innerHTML = headerText;

            // Kick the countdowns for the freshly injected rows
            if (typeof startCountdownTicker === 'function') {
                startCountdownTicker();
            } else if (typeof updateAllCountdowns === 'function') {
                updateAllCountdowns();
            }

            bootstrap.Modal.getOrCreateInstance(document.getElementById('ptiModal')).show();
        } catch (err) {
            // --- 5. Error Handling ---
            console.error('[badge click] failed', err);
            document.getElementById('ptiPayload').innerHTML =
                `<div class="text-danger small">Failed to load data: ${esc(err.message || err)}</div>`;
            bootstrap.Modal.getOrCreateInstance(document.getElementById('ptiModal')).show();
        }
    });


    // copy the SDP text from the modal body to the clipboard
    function copySdpFromModal(){
        const text = document.getElementById('sdpModalBody').textContent || '';
        navigator.clipboard.writeText(text).catch(()=>{});
    }

    /* ---------- Paging & fetching ---------- */
    let nextOffset = 0, loading = false, hasMore = true;

    // build URLSearchParams for the current query and facet filters
    function currentFilters(){
        const q = (document.getElementById('searchInput')?.value || '').trim();
        filterState.q = q;
        const params = new URLSearchParams();
        params.set('offset', String(nextOffset));
        params.set('limit',  String(PAGE_SIZE));
        if (q) params.set('q', q);
        params.set('inst',  'ibc');
        //(filterState.inst||[]).forEach(v => params.append('inst[]', v));
        //(filterState.rp||[]).forEach(v   => params.append('rp[]', v));
        return params;
    }

    // fetch the next page of rows, append to the table, hydrate names, and update paging
    async function fetchPage() {
        if (loading || !hasMore) return;

        loading = true;
        const spinner = document.getElementById('load-more-spinner');
        if (spinner) spinner.style.display = 'inline';

        try {
            const url = `${API.SERVICES_PAGE}?${buildFilterParams().toString()}`;
            const res = await fetch(url, { cache: 'no-store' });
            if (!res.ok) throw new Error('HTTP ' + res.status);

            const payload = await res.json();
            const tbody = document.getElementById('connections-body');

            for (const row of payload.rows) {
                const id = `r-${row.booking_id}-${row.booking_version}`;
                if (!document.getElementById(id)) {
                    tbody.insertAdjacentHTML('beforeend', renderRowHtml(row));
                }
            }

            await resolveDeviceBadges(tbody);
            refreshVisibleCounts();

            nextOffset = (payload.nextOffset != null)
                ? payload.nextOffset
                : (nextOffset + payload.rows.length);
            hasMore = !!payload.hasMore;

            updateRecordCount();
            console.log('[PAGING] got', { rows: payload.rows.length, nextOffset, hasMore });
        } catch (e) {
            console.warn('[PAGING] failed', e);
        } finally {
            if (spinner) spinner.style.display = 'none';
            const btn = document.getElementById('load-more-btn');
            if (btn) btn.style.display = hasMore ? 'inline-block' : 'none';
            loading = false;
        }
    }




    /* ---------- Count refreshers ---------- */
    function collectVisibleIds(){
        const rows = Array.from(document.querySelectorAll('#connections-body tr'));
        const devs = new Set(), senders = new Set();
        for (const tr of rows){
            const f = tr.getAttribute('data-from-id')||'';
            const t = tr.getAttribute('data-to-id')||'';
            const s = tr.getAttribute('data-sender-id')||'';
            if (f) devs.add(f); if (t) devs.add(t); if (s) senders.add(s);
        }
        return { devices:[...devs], senders:[...senders] };
    }

    async function refreshDeviceCounts(ids){
        const uniq = [...new Set((ids||[]).map(s => String(s||'').trim()).filter(Boolean))];
        if (!uniq.length) return;

        const url = new URL(API.DEVICE_COUNTS, window.location.href);
        url.searchParams.set('ids', uniq.join(','));

        try {
            const r = await fetch(url.toString(), { cache: 'no-store' });
            if (!r.ok) return;
            const p = await r.json();
            if (!p.ok) return;
            const map = p.counts || {};

            document.querySelectorAll('#connections-body tr').forEach(tr => {
                const f = tr.getAttribute('data-from-id') || '';
                const t = tr.getAttribute('data-to-id')   || '';

                if (f && map[f]) {
                    const b = tr.querySelector('[data-badge="from"]');
                    if (b) { b.textContent = String(map[f].from_count ?? 0); b.style.display = 'inline-flex'; }
                }
                if (t && map[t]) {
                    const b = tr.querySelector('[data-badge="to"]');
                    if (b) { b.textContent = String(map[t].to_count ?? 0); b.style.display = 'inline-flex'; }
                }
            });
        } catch {}
    }

    async function refreshSenderCounts(ids){
        if (!ids.length) return;
        const url = new URL(API.SENDER_COUNTS,  window.location.href);
        url.searchParams.set('ids', ids.join(','));
        try{
            const r = await fetch(url.toString(), {cache:'no-store'}); if(!r.ok) return;
            const p = await r.json(); if(!p.ok) return;
            const map = p.counts || {};
            document.querySelectorAll('#connections-body tr').forEach(tr=>{
                const s = tr.getAttribute('data-sender-id')||'';
                if (s && (s in map)){
                    const b = tr.querySelector('[data-badge="sender"]'); if (b){ b.textContent = String(map[s] ?? 0); b.style.display='inline-flex'; }
                }
            });
        }catch{}
    }
    function refreshVisibleCounts(){
        const {devices, senders} = collectVisibleIds();
        queueDeviceCountRefresh(devices);
        queueSenderCountRefresh(senders);
    }

    /* ---------- Row upsert/remove for WS ---------- */
    function updateRecordCount(){
        const c = document.querySelectorAll('#connections-body tr').length;
        const el = document.getElementById('record-count'); if (el) el.textContent = `${c.toLocaleString()} record${c!==1?'s':''}`;
    }
    async function upsertRow(row){
        const id = `r-${row.booking_id}-${row.booking_version}`;
        const html = renderRowHtml(row);
        const tbody = document.getElementById('connections-body');
        const existing = document.getElementById(id);
        if (existing) existing.outerHTML = html; else tbody.insertAdjacentHTML('afterbegin', html);
        updateRecordCount();

        // 🔧 resolve names + turn into links (if allowed)
        const tr = document.getElementById(id);
        await resolveDeviceBadges(tr);
    }

    function removeRowByKey(bid,bver){
        const id = `r-${bid}-${bver}`;
        const tr = document.getElementById(id);
        if (!tr) return;
        const f = tr.getAttribute('data-from-id')||'';
        const t = tr.getAttribute('data-to-id')||'';
        const s = tr.getAttribute('data-sender-id')||'';
        tr.remove();
        updateRecordCount();
        // refresh only affected
        queueDeviceCountRefresh([f,t].filter(Boolean));
        queueSenderCountRefresh([s].filter(Boolean));
    }

    // build URLSearchParams for the current query and facet filters
    function buildFilterParams(options = {}) {
        const { forExport = false } = options;

        const q = (document.getElementById('searchInput')?.value || '').trim();
        filterState.q = q;

        const params = new URLSearchParams();

        if (!forExport) {
            // normal paged mode
            params.set('offset', String(nextOffset));
            params.set('limit',  String(PAGE_SIZE));
        } else {
            // export mode → all matching rows
            params.set('offset', '0');
            params.set('limit',  '0');      // convention: 0 = no limit
        }

        if (q) params.set('q', q);

        // Use the *same* inst/rp state as the grid
        (filterState.inst || []).forEach(v => params.append('inst[]', v));
        (filterState.rp   || []).forEach(v => params.append('rp[]', v));

        return params;
    }


    /* ---------- WebSockets ---------- */
    function startWS(url, dotEl, onMessage){
        let ws, retry=0;
        const connect = () => {
            setDot(dotEl,'amber');
            ws = new WebSocket(url);
            ws.onopen = ()=>{ retry=0; setDot(dotEl,'green'); };
            ws.onclose= ()=>{ setDot(dotEl,'red'); const backoff=Math.min(1000*Math.pow(2,retry++),15000); setTimeout(connect, backoff); };
            ws.onerror= e=>console.warn('[WS error]', e);
            ws.onmessage = ev => {
                try{
                    const arr = JSON.parse(ev.data);
                    if (Array.isArray(arr)) onMessage(arr);
                }catch(e){ console.warn('[WS bad JSON]', e); }
            };
        };
        connect();
        return () => { try{ ws && ws.close(); }catch{} };
    }

    /**
     * Updates a status badge's text and color class.
     * @param {string} badgeId - The ID of the HTML badge element.
     * @param {string} status - The status string (e.g., 'ok', 'error', 'connecting').
     */
    function updateStatusBadge(badgeId, status) {
        const badgeElement = document.getElementById(badgeId);
        if (!badgeElement) {
            console.error(`Badge element with ID "${badgeId}" not found.`);
            return;
        }

        // Reset all previous Bootstrap color classes (bg-*)
        badgeElement.className = badgeElement.className.replace(/\b(bg-success|bg-danger|bg-warning|bg-secondary)\b/g, '');

        const badgeText = status.toUpperCase();
        let badgeClass = 'bg-secondary';

        // Map the status to a Bootstrap color class
        switch (status.toLowerCase()) {
            case 'ok':
            case 'connected':
                badgeClass = 'bg-success';
                break;
            case 'Error':
            case 'disconnected':
            case 'critical':
                badgeClass = 'bg-danger';
                break;
            case 'Starting':
                badgeClass = 'bg-info';
                break;
        }

        // Apply the new class and update the text
        badgeElement.classList.add(badgeClass);
        badgeElement.textContent = badgeText;
    }

    /**
     * Updates the background color class of the CONNECTION STATUS badge
     * while preserving the badge's text (LDC or NHN).
     */
    function updateConnectionStatus(elementId, status) {
        const element = document.getElementById(elementId);
        if (!element) return;

        const currentClasses = element.className.replace(/\b(text-bg-success|text-bg-danger|text-bg-warning|text-bg-primary|text-bg-secondary|text-bg-info)\b/g, '');
        let newClass = 'text-bg-secondary';

        switch (status.toLowerCase()) {
            case 'connected':
            case 'open':
                newClass = 'text-bg-success';
                break;
            case 'connecting':
                newClass = 'text-bg-warning';
                break;
            case 'closed':
            case 'error':
                newClass = 'text-bg-danger';
                break;
            default:
                newClass = 'text-bg-secondary';
                break;
        }
        element.className = `${currentClasses} ${newClass}`;
        // Text is intentionally not modified here, remains LDC/NHN
    }

    /**
     * Updates a specific Bootstrap badge's text and color class.
     * @param {string} badgeId - The ID of the HTML badge element.
     * @param {string} status - The status string (e.g., 'ok', 'error', 'connecting').
     */
    function updateBadge(badgeId, status) {
        const badgeElement = document.getElementById(badgeId);
        if (!badgeElement) {
            console.error(`Badge element with ID "${badgeId}" not found.`);
            return;
        }

        badgeElement.className = badgeElement.className.replace(/\b(text-bg-success|text-bg-danger|text-bg-warning|text-bg-secondary)\b/g, '');

        const badgeText = status.toUpperCase();
        let badgeClass = 'text-bg-secondary';



        switch (status.toLowerCase()) {

            case 'ok':
            case 'connected':
                badgeClass = 'text-bg-success';
                break;

            case 'warning':
                badgeClass = 'text-bg-warning';
                break;

            case 'pending':
            case 'connecting':
            case 'starting':
            case 'stopping':
                badgeClass = 'text-bg-info';
                break;

            case 'disconnected':
            case 'critical':
            case 'fatal':
            case 'error':
            case 'stopped':

            case 'broken':
            default:
                badgeClass = 'text-bg-danger';
                break;
        }

        badgeElement.classList.add(badgeClass);
        badgeElement.textContent = badgeText;
    }

    // GLOBAL HEARTBEAT MANAGER
    let heartbeatTimer = null;
    let heartbeatIntervalMs = 60000; // default 60 seconds

    function restartHeartbeatTimer(newInterval) {
        if (newInterval > 0)
        {
            heartbeatIntervalMs = newInterval * 2.05;
        }

        if (heartbeatTimer) clearTimeout(heartbeatTimer);

        heartbeatTimer = setTimeout(() => {
            console.warn("[HEARTBEAT] No heartbeat received within interval → triggering overlay");
            showOverlayAndStartTimer();
        }, heartbeatIntervalMs);
    }


    function showOverlayAndStartTimer() {
        const overlay = document.getElementById('fullPageOverlay');
        const tsEl = document.getElementById('overlayTimestamp');
        const cdEl = document.getElementById('overlayCountdown');

        if (!overlay || !tsEl || !cdEl) {
            console.warn("[OVERLAY] Missing elements, cannot show overlay.");
            return;
        }

        console.warn("[OVERLAY] SHOW OVERLAY");

        // Stop old countdown
        if (overlayCountdownTimer) clearInterval(overlayCountdownTimer);

        // overlay.style.display = 'block';
        // overlay.classList.add('show');

        const now = new Date();
        tsEl.textContent = now.toLocaleTimeString("en-GB") + " on " + now.toLocaleDateString("en-GB");

        let countdown = Math.ceil(RECONNECT_DELAY_MS / 1000);

        const tick = () => {
            cdEl.textContent = countdown;
            countdown--;

            if (countdown < 0) {
                clearInterval(overlayCountdownTimer);
                cdEl.textContent = "... Retrying ...";
            }
        };

        tick();
        overlayCountdownTimer = setInterval(tick, 1000);
    }

    function hideOverlay() {
        console.warn("[OVERLAY] HIDE OVERLAY");

        const overlay = document.getElementById('fullPageOverlay');
        if (!overlay) return;

        overlay.style.display = 'none';
        overlay.classList.remove('show');

        if (overlayCountdownTimer) clearInterval(overlayCountdownTimer);
        overlayCountdownTimer = null;
    }

    function setupWebSocket(name, uri, badgeId, connectionId) {
        console.log(`[${name}] Connecting to`, uri);

        updateConnectionStatus(connectionId, 'connecting');

        const ws = new WebSocket(uri);

        ws.onopen = () => {
            console.log(`[${name}] WebSocket OPEN`);
            updateConnectionStatus(connectionId, 'connected');
            updateBadge(badgeId, 'pending');
            // add a connecting... message to strap
            hideOverlay();
        };

        ws.onerror = (err) => {
            console.error(`[${name}] WebSocket ERROR`, err);
            updateConnectionStatus(connectionId, 'error');
            updateBadge(badgeId, 'critical');

            showOverlayAndStartTimer();
        };

        ws.onclose = (ev) => {
            console.warn(`[${name}] WebSocket CLOSED (code ${ev.code})`);
            updateConnectionStatus(connectionId, 'closed');
            updateBadge(badgeId, 'critical');

            showOverlayAndStartTimer();

            setTimeout(() => setupWebSocket(name, uri, badgeId, connectionId), RECONNECT_DELAY_MS);
        };

        ws.onmessage = (event) => {
            let data;
            try {
                data = JSON.parse(event.data);
            } catch (err) {
                console.error(`[${name}] Invalid JSON`, err);
                return;
            }

            const status = data.status;
            const status_message = data.status_message;
            const interval = data.interval;

            console.log(`[${name}] heartbeat`, data);
            // Restart heartbeat timer each heartbeat
            restartHeartbeatTimer(interval);

            // CRITICAL → overlay
            if (['error','lost_connection'].includes(status)) {
                console.warn(`[${name}] CRITICAL STATUS → overlay`);
                showOverlayAndStartTimer();
            } else {
                // 'starting', 'connecting', 'gathering_data', 'running'

                // OK → clear overlay
                hideOverlay();
            }

            updateBadge(badgeId, status);
        };
    }


    /* ---------- Boot ---------- */
    document.addEventListener('DOMContentLoaded', async ()=> {
        if (window.__liveBooted) return;   // <- prevents double init
        window.__liveBooted = true;

        const versionElement = document.getElementById('data-site-version');
        if (versionElement) {
            versionElement.title = esc($siteVersion);
        }

        startCountdownTicker();
        // Facets + saved filters
        await loadDistinctsAndFacets();

        // Search box: Enter triggers reset+reload
        const search = document.getElementById('searchInput');
        search?.addEventListener('keydown', e => {
            if (e.key === 'Enter') {
                nextOffset = 0;
                hasMore = true;
                document.getElementById('connections-body').innerHTML = '';
                fetchPage();
            }
        });

        // Paging: sentinel & button
        const btn = document.getElementById('load-more-btn');
        btn.addEventListener('click', () => fetchPage());
        const sentinel = document.getElementById('sentinel');
        const io = new IntersectionObserver((entries) => {
            if (entries[0].isIntersecting && !loading && hasMore) fetchPage();
        }, {root: document.querySelector('.table-scroller'), threshold: 0, rootMargin: '0px 0px 200px 0px'});
        io.observe(sentinel);

        // Initial page
        await fetchPage();

        // WS wiring
        const dotUpdate = document.getElementById('dot-update');
        const dotRemove = document.getElementById('dot-remove');

        window.stopUpdate = startWS(WS_UPDATE, dotUpdate, async arr => {
            updateWsTimestamp();

            const devsToRefresh = new Set();

            for (const it of arr) {
                const id = String(it.ID ?? it.Id ?? it.id ?? '');
                const ver = String(it.Version ?? it.version ?? it.Ver ?? '');
                if (!id || !ver) continue;

                try {
                    const url = new URL(API.SERVICES_ROW, window.location.href);
                    url.searchParams.set('booking_id', id);
                    url.searchParams.set('booking_version', ver);
                    const r = await fetch(url.toString(), {cache: 'no-store'});
                    if (!r.ok) continue;
                    const p = await r.json();
                    if (!p.present) continue;

                    const row = p.data || {};
                    upsertRow(row);

                    // collect affected device ids (dedup automatically by Set)
                    if (row.from_device_id) devsToRefresh.add(row.from_device_id);
                    if (row.to_device_id) devsToRefresh.add(row.to_device_id);

                    // sender counts can also be queued similarly if needed
                    queueSenderCountRefresh([row.from_ipath_logical_source_id]); // optional if you add same batching for senders
                } catch {
                }
            }

            queueDeviceCountRefresh([...devsToRefresh]);
        });


        window.stopRemove = startWS(WS_REMOVE, dotRemove, arr => {
            updateWsTimestamp();
            for (const it of arr) {
                const id = String(it.ID ?? it.Id ?? it.id ?? '');
                const ver = String(it.Version ?? it.version ?? it.Ver ?? '');
                if (!id || !ver) continue;
                removeRowByKey(id, ver);
            }
        });

        addEventListener('click', (e) => {
            const trigger = e.target.closest('.popover-launch');
            if (!trigger) return;

            // Build content with simple quotes to avoid stray backtick issues
            const content =
                '<div class="d-grid gap-1">' +
                '<button type="button" class="btn btn-sm btn-outline-secondary" data-pop="tech">Tech Info</button>' +
                '<button type="button" class="btn btn-sm btn-outline-secondary" data-pop="history">History</button>' +
                '</div>';

            // Dispose any existing instance bound to this trigger
            const existing = bootstrap.Popover.getInstance(trigger);
            if (existing) existing.dispose();

            new bootstrap.Popover(trigger, {
                container: 'body',
                html: true,
                sanitize: false,     // <-- important
                customClass: 'mini-popover',
                content: content,
                placement: 'right',
                trigger: 'focus'
            }).show();
        });


        document.addEventListener('click', async (e) => {
            // Only clicks inside an open popover
            const pop = e.target.closest('.popover');
            if (!pop) return;

            // Find the trigger that created this popover
            const popId = pop.getAttribute('id');
            const trigger = popId ? document.querySelector('[aria-describedby="' + popId + '"]') : null;
            if (!trigger || !trigger.classList.contains('popover-launch')) return;

            document.addEventListener('click', async (e) => {
                // Only clicks inside an open popover
                const pop = e.target.closest('.popover');
                if (!pop) return;

                // Find the trigger that created this popover
                const popId = pop.getAttribute('id');
                const trigger = popId ? document.querySelector('[aria-describedby="' + popId + '"]') : null;
                if (!trigger || !trigger.classList.contains('popover-launch')) return;

                if (e.target.matches('[data-pop="tech"]')) {
                    const tr = trigger.closest('tr');
                    const fromTxt    = b64decUtf8(tr?.dataset.fromB64 || '');
                    const toTxt      = b64decUtf8(tr?.dataset.toB64   || '');
                    const detailsTxt = b64decUtf8(tr?.dataset.descB64 || '');

                    const fromObj    = parseOnceOrTwice(fromTxt);
                    const toObj      = parseOnceOrTwice(toTxt);
                    let   detailsObj = parseOnceOrTwice(detailsTxt);

                    // 🔎 If we have tags.workstation as an integer (or numeric string), resolve and rewrite
                    try {
                        const rawWs = detailsObj?.tags?.workstation;
                        const wsId  = Number.isInteger(rawWs)
                            ? rawWs
                            : Number.parseInt(String(rawWs ?? ''), 10);

                        if (Number.isFinite(wsId)) {
                            const wsData = await resolveWorkstationData(wsId);
                            const label  = wsData
                                ? (wsData.short_name || wsData.name || 'Unknown')
                                : 'Unknown';

                            // Ensure structure exists, then overwrite the value with "3930 (label)"
                            if (!detailsObj || typeof detailsObj !== 'object') detailsObj = {};
                            if (!detailsObj.tags || typeof detailsObj.tags !== 'object') detailsObj.tags = {};

                            detailsObj.tags.workstation = `${wsId} (${label})`;
                        }
                    } catch (err) {
                        console.warn('[workstation enrichment] failed', err);
                        // leave detailsObj as-is if anything goes wrong
                    }

                    // Pretty JSON (stable key order)
                    const prettyFrom    = JSON.stringify(sortDeep(fromObj),    null, 2);
                    const prettyTo      = JSON.stringify(sortDeep(toObj),      null, 2);
                    const prettyDetails = JSON.stringify(sortDeep(detailsObj), null, 2);

                    // Inject
                    const fromEl    = document.getElementById('techFrom');
                    const toEl      = document.getElementById('techTo');
                    const detailsEl = document.getElementById('techDetails');

                    fromEl.textContent    = prettyFrom;
                    toEl.textContent      = prettyTo;
                    detailsEl.textContent = prettyDetails;

                    const placeholder = document.getElementById('data-descriptor-label');
                    if (placeholder) {
                        placeholder.textContent = tr?.dataset.descriptorLabel || '';
                    }

                    bootstrap.Modal.getOrCreateInstance(document.getElementById('techInfoModal')).show();
                }

                if (e.target.matches('[data-pop="history"]')) {
                    bootstrap.Modal.getOrCreateInstance(document.getElementById('historyModal')).show();
                }
            });

            if (e.target.matches('[data-pop="history"]')) {
                document.getElementById('techFrom'); // (your original no-op)
                bootstrap.Modal.getOrCreateInstance(document.getElementById('historyModal')).show();
            }
        });

        const exportBtn = document.getElementById('exportCsvBtn');
        if (exportBtn) {
            exportBtn.addEventListener('click', () => {
                console.log("click");
                const params = buildFilterParams({ forExport: true });
                // Hit the CSV export endpoint – browser download
                window.location = 'api/export_ipath_services_csv.php?' + params.toString();
            });
        }



        setupWebSocket("ipath_ldc", WS_STATUS_LDC, BADGE_STATUS_LDC, BADGE_WS_STATUS_LDC);
//        setupWebSocket("ipath_nhn", WS_STATUS_NHN, BADGE_STATUS_NHN, BADGE_WS_STATUS_NHN);


    });


</script>
</body>
</html>
